﻿#region Using Statements

using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.Mvc;
using System.Web.Routing;

using HIPS.PcehrDataStore.Schemas.Enumerators;

using HIPS.Web.ModelInterface;
using HIPS.Web.ModelInterface.Common;
using HIPS.Web.ModelInterface.PcehrView;
using HIPS.Web.Model.Common;

using HIPS.Web.Data.Hips;
using HIPS.Web.Data.Hips.PcehrView;
using HIPS.Web.Data.WebsiteDb;

using HIPS.Web.Components.Cache;

using HIPS.Web.UI.Controllers;
using HIPS.Web.UI.Helpers;
using HIPS.Web.UI.ViewModels.PcehrView;

#endregion

namespace HIPS.Web.UI.Filters
{

    /// <summary>
    /// Provides an action filter that supports checking access to the PCEHR.
    /// </summary>
    /// <history>
    ///   <change user="David Sampson (Chamonix)" date="16 January 2014">Initial version.</change>
    /// </history>
    public class PcehrAccessFilter : IActionFilter
    {

        #region Fields

        /// <summary>
        /// List of application-specific settings.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="16 January 2014">Initial version.</change>
        /// </history>
        private readonly List<Setting> Settings;

        /// <summary>
        /// Gets the patient repository to be used by this class.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="16 January 2014">Initial version.</change>
        /// </history>
        private readonly IPatientRepository PatientRepository;

        /// <summary>
        /// Gets the PCEHR view repository to be used by this class.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="16 January 2014">Initial version.</change>
        /// </history>
        private readonly IPcehrViewRepository PcehrViewRepository;

        #endregion

        #region Properties

        /// <summary>
        /// Gets the default hospital code system used by the application.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="16 January 2014">Initial version.</change>
        /// </history>
        private string DefaultHospitalCodeSystem
        {
            get
            {
                return Settings.GetSettingValue(Setting.SettingCodes.DefaultHospitalCodeSystem);
                //return "pasFacCd";
            }
        }

        /// <summary>
        /// Gets or sets a Boolean value that indicates whether to automatically attempt to gain access if required.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="16 January 2014">Initial version.</change>
        /// </history>
        private bool AutoAccess { get; set; }

        #endregion

        #region Constructors

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="patientRepository">Patient repository to be used by this class.</param>
        /// <param name="pcehrViewRepository">PCEHR view repository to be used by this class.</param>
        /// <param name="settingsRepository">Settings repository to be used by this class.</param>
        /// <param name="autoAccess">A Boolean value that indicates whether to automatically attempt to gain access if required. Defaults to true.</param>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="16 January 2014">Initial version.</change>
        /// </history>
        public PcehrAccessFilter(IPatientRepository patientRepository, IPcehrViewRepository pcehrViewRepository, ISettingsRepository settingsRepository, bool autoAccess = true)
        {
            this.PatientRepository = patientRepository;
            this.PcehrViewRepository = pcehrViewRepository;
            using (settingsRepository)
            {
                this.Settings = settingsRepository.GetSettings();
            }
            this.AutoAccess = autoAccess;
        }

        #endregion

        #region Methods

        #region Overrides

        /// <summary>
        /// Overrides the OnActionExecuting method to implement the action filter logic.
        /// </summary>
        /// <param name="filterContext"></param>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="16 January 2014">Initial version.</change>
        /// </history>
        public void OnActionExecuting(ActionExecutingContext filterContext)
        {
            //Retrieve required data from context.
            PcehrViewModelBase request = null;
            string hospitalId = null;
            string patientId = null;
            string requestedResource = "";

            //Support different types of parameters:
            //PD View action uses model binding and a single "request" ViewModel parameter.
            //Other actions use individual parameters.

            if (filterContext.ActionParameters.ContainsKey("request"))
            {
                request = (PcehrViewModelBase)filterContext.ActionParameters["request"];
                hospitalId = request.HospitalId;
                patientId = request.PatientId;
            }
            if (filterContext.ActionParameters.ContainsKey("hospitalId") && filterContext.ActionParameters.ContainsKey("patientId"))
            {
                hospitalId = (string)filterContext.ActionParameters["hospitalId"];
                patientId = (string)filterContext.ActionParameters["patientId"];
            }
            if (string.IsNullOrEmpty(hospitalId))
            {
                throw new ArgumentNullException("Expected parameter 'hospitalId' not found.", "hospitalId");
            }
            if (string.IsNullOrEmpty(patientId))
            {
                throw new ArgumentNullException("Expected parameter 'patientId' not found.", "patientId");
            }

            if ((filterContext.HttpContext != null) && (filterContext.HttpContext.Request != null))
            {
                requestedResource = filterContext.HttpContext.Request.Url.AbsolutePath;
            }
            var patientIdentifier = new CommonSchemas.PatientIdentifier.Mrn(patientId, hospitalId, this.DefaultHospitalCodeSystem);

            //Check access:
            PcehrAccessManager accessMgr = new PcehrAccessManager(this.PatientRepository, this.PcehrViewRepository);

            var accessResult = accessMgr.AllowAccess(patientIdentifier, this.AutoAccess, ((Controller)filterContext.Controller).GetCurrentUserDetails());

            if (! accessResult.AccessAllowed)
            {
                filterContext.Controller.TempData["PcehrAccessMessage"] = accessResult.AccessMessage;

                //Redirect to "Gain Access" action.
                filterContext.Result = new RedirectToRouteResult
                    (
                        new RouteValueDictionary()
                        {
                            { "action", "GainAccess" },
                            { "controller", "PcehrView" },
                            { "hospitalId", hospitalId },
                            { "patientId", patientId },
                            { "returnUrl", requestedResource }
                        }
                    );
            }

            //Otherwise, proceed to originally requested function.
        }

        /// <summary>
        /// Overrides the OnActionExecuted method to implement the action filter logic.
        /// </summary>
        /// <param name="filterContext"></param>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="16 January 2014">Initial version.</change>
        /// </history>
        public void OnActionExecuted(ActionExecutedContext filterContext)
        {
            //Do nothing.
        }

        #endregion

        #endregion

    }

}